<?php
/**
 * NewsSync Main Admin Layout
 * Sidebar navigation + content wrapper structure
 *
 * @package NewsSync
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Check for success messages
$newssync_settings_saved = get_transient( 'newssync_settings_saved' );
$newssync_cache_cleared = get_transient( 'newssync_cache_cleared' );

if ( $newssync_settings_saved ) {
    delete_transient( 'newssync_settings_saved' );
}
if ( $newssync_cache_cleared ) {
    delete_transient( 'newssync_cache_cleared' );
}

// Language and locale config
$newssync_languages_dir = NEWSSYNC_PLUGIN_DIR . 'languages';
$newssync_current_lang = get_option( 'newssync_plugin_language', 'en_US' );
$newssync_available_locales = array(
    'en_US' => 'English (US)',
    'pt_PT' => 'Português (PT)',
    'pt_BR' => 'Português (BR)',
    'es_ES' => 'Español (ES)',
    'de_DE' => 'Deutsch (DE)',
    'it_IT' => 'Italian (IT)',
    'fr_FR' => 'French (FR)',
);

// Check if a specific section should be forced active (set by submenu pages)
global $newssync_force_section;
$newssync_forced_section = !empty( $newssync_force_section ) ? $newssync_force_section : '';

// Determine active tab (for backward compatibility)
// If forced section exists, use it; otherwise check GET parameter
if ( $newssync_forced_section ) {
    $newssync_active_tab = $newssync_forced_section;
} else {
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab parameter is for navigation only, not form processing
    $newssync_active_tab = function_exists( 'newssync_request_string' ) ? newssync_request_string( 'tab', 'GET', 'hub' ) : ( isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'hub' );
    $newssync_active_tab = sanitize_text_field( $newssync_active_tab );
}

// Tabs for old navigation (hidden by CSS)
$newssync_tabs = array(
    'feeds'  => __( 'Feeds', 'rss-newssync' ),
    'settings' => __( 'Settings', 'rss-newssync' ),
);
?>

<?php if ( $newssync_forced_section ) : ?>
<script>
    // Force activate a specific section (set by submenu callback)
    window.newssyncForceSection = '<?php echo esc_js( $newssync_forced_section ); ?>';
    console.log('NewsSync: Forcing section:', window.newssyncForceSection);
</script>
<?php endif; ?>

<script>console.log('NewsSync: layout - before wrapper (server render)');</script>

<!-- Main Admin Wrapper with Sidebar -->
<div class="newssync-admin-wrap">

    <script>console.log('NewsSync: layout - after wrapper open (server render)');</script>

    <!-- Sidebar Navigation -->
    <aside class="newssync-sidebar">
        <div class="newssync-sidebar-brand">
            <div class="newssync-brand-icon">N</div>
            <h2 class="newssync-brand-text"><?php esc_html_e( 'NewsSync', 'rss-newssync' ); ?></h2>
        </div>

        <nav>
            <ul>
                <li>
                    <a href="#hub" data-section="hub" class="<?php echo $newssync_active_tab === 'hub' ? 'active' : ''; ?>">
                        <span class="dashicons dashicons-admin-home"></span>
                        <?php esc_html_e( 'Hub', 'rss-newssync' ); ?>
                    </a>
                </li>
                <li>
                    <a href="#feeds" data-section="feeds" class="<?php echo $newssync_active_tab === 'feeds' ? 'active' : ''; ?>">
                        <span class="dashicons dashicons-rss"></span>
                        <?php esc_html_e( 'Feeds', 'rss-newssync' ); ?>
                    </a>
                </li>
                <li>
                    <a href="#settings" data-section="settings" class="<?php echo $newssync_active_tab === 'settings' ? 'active' : ''; ?>">
                        <span class="dashicons dashicons-admin-generic"></span>
                        <?php esc_html_e( 'Settings', 'rss-newssync' ); ?>
                    </a>
                </li>
                <li>
                    <a href="#tools" data-section="tools" class="<?php echo $newssync_active_tab === 'tools' ? 'active' : ''; ?>">
                        <span class="dashicons dashicons-admin-tools"></span>
                        <?php esc_html_e( 'Tools', 'rss-newssync' ); ?>
                    </a>
                </li>
                <li>
                    <a href="#documentation" data-section="documentation" class="<?php echo $newssync_active_tab === 'documentation' ? 'active' : ''; ?>">
                        <span class="dashicons dashicons-book"></span>
                        <?php esc_html_e( 'Documentation', 'rss-newssync' ); ?>
                    </a>
                </li>
            </ul>
        </nav>
    </aside>

    <!-- Content Area -->
    <main class="newssync-content">

        <!-- Content Header -->
        <header class="newssync-content-header">
            <div class="newssync-content-header-left">
                <div class="newssync-content-header-icon">
                    <span class="dashicons dashicons-admin-home"></span>
                </div>
                <div class="newssync-content-header-title">
                    <?php
                    $newssync_titles = array(
                        'hub' => esc_html__( 'Hub', 'rss-newssync' ),
                        'feeds' => esc_html__( 'Feeds', 'rss-newssync' ),
                        'settings' => esc_html__( 'Settings', 'rss-newssync' ),
                        'tools' => esc_html__( 'System Information', 'rss-newssync' ),
                        'documentation' => esc_html__( 'Documentation', 'rss-newssync' ),
                    );
                    $newssync_subtitles = array(
                        'hub' => esc_html__( 'Dashboard and quick stats', 'rss-newssync' ),
                        'feeds' => esc_html__( 'Manage your RSS feeds', 'rss-newssync' ),
                        'settings' => esc_html__( 'Configure plugin settings', 'rss-newssync' ),
                        'tools' => esc_html__( 'Database status and optimization tools', 'rss-newssync' ),
                        'documentation' => esc_html__( 'Learn how to use all shortcode parameters', 'rss-newssync' ),
                    );
                    $newssync_current = isset( $newssync_titles[ $newssync_active_tab ] ) ? $newssync_active_tab : 'hub';
                    ?>
                    <h1><?php echo esc_html( $newssync_titles[ $newssync_current ] ); ?></h1>
                    <p class="newssync-content-header-subtitle"><?php echo esc_html( $newssync_subtitles[ $newssync_current ] ); ?></p>
                </div>
            </div>
            <div class="newssync-content-header-actions">
                <!-- Language selector -->
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline; margin:0;">
                    <?php wp_nonce_field( 'newssync_save_language', 'newssync_lang_nonce' ); ?>
                    <input type="hidden" name="action" value="newssync_save_language">
                    <input type="hidden" name="newssync_current_tab" value="<?php echo esc_attr( $newssync_active_tab ); ?>">
                    <select name="newssync_plugin_language" onchange="this.form.submit()" style="padding:8px 12px; border-radius:6px; border:1px solid #CBD5E1;">
                    <?php
                    if ( is_dir( $newssync_languages_dir ) ) {
                        $newssync_found = false;
                        foreach ( $newssync_available_locales as $newssync_locale => $newssync_name ) {
                            $newssync_mo_file = $newssync_languages_dir . "/rss-newssync-{$newssync_locale}.mo";
                            if ( file_exists( $newssync_mo_file ) ) {
                                $newssync_found = true;
                                echo '<option value="' . esc_attr( $newssync_locale ) . '" ' . selected( $newssync_current_lang, $newssync_locale, false ) . '>' . esc_html( $newssync_name ) . '</option>';
                            }
                        }
                        if ( ! $newssync_found ) {
                            echo '<option value="en_US">English</option>';
                        }
                    } else {
                        echo '<option value="en_US">English</option>';
                    }
                    ?>
                    </select>
                </form>

                <!-- Clear Cache button -->
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline;">
                    <?php wp_nonce_field( 'newssync_clear_cache_action', 'newssync_clear_cache_nonce' ); ?>
                    <input type="hidden" name="action" value="newssync_clear_cache">
                    <input type="hidden" name="newssync_current_tab" value="<?php echo esc_attr( $newssync_active_tab ); ?>">
                    <button type="submit" class="button newssync-btn-secondary">
                        <span class="dashicons dashicons-trash"></span>
                        <?php esc_html_e( 'Clear Cache', 'rss-newssync' ); ?>
                    </button>
                </form>
            </div>
        </header>

        <!-- Content Body -->
        <div class="newssync-content-body">

            <?php if ( $newssync_settings_saved ) : ?>
                <div class="notice notice-success is-dismissible" style="margin: 0 0 20px 0;">
                    <p><?php esc_html_e( 'Settings saved successfully!', 'rss-newssync' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( $newssync_cache_cleared ) : ?>
                <div class="notice notice-success is-dismissible" style="margin: 0 0 20px 0;">
                    <p><?php esc_html_e( 'Cache cleared successfully!', 'rss-newssync' ); ?></p>
                </div>
            <?php endif; ?>

            <!-- Old tabs wrapper (hidden by CSS, kept for compatibility) -->
            <h2 class="nav-tab-wrapper">
                <?php foreach ( $newssync_tabs as $newssync_tab => $newssync_name ) :
                    $newssync_page_slug = 'rss-newssync';
                    $newssync_tab_url = admin_url( 'admin.php?page=' . $newssync_page_slug . '&tab=' . $newssync_tab );
                ?>
                    <a href="<?php echo esc_url( $newssync_tab_url ); ?>" class="nav-tab <?php echo $newssync_active_tab === $newssync_tab ? 'nav-tab-active' : ''; ?>"><?php echo esc_html( $newssync_name ); ?></a>
                <?php endforeach; ?>
            </h2>

            <!-- Include all sections -->
            <?php
            // Force refresh: 2026-01-16 17:30:00
            require_once __DIR__ . '/../sections/newssync-hub-section.php';
            require_once __DIR__ . '/../sections/newssync-feeds-section.php';
            require_once __DIR__ . '/../sections/newssync-settings-section.php';
            require_once __DIR__ . '/../sections/newssync-tools-section.php';
            require_once __DIR__ . '/../sections/newssync-docs-section.php';
            ?>

        </div><!-- .newssync-content-body -->
    </main><!-- .newssync-content -->
</div><!-- .newssync-admin-wrap -->
