<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Funções otimizadas para criar posts a partir de items RSS.
 *
 * RESOLUÇÕES:
 * - Todas as queries usam cache agressivo
 * - Lookups via funções indexadas (newssync_find_post_id_by_guid)
 * - ZERO meta_query diretas
 * - Fallback automático se tabela não existir
 */

if ( ! function_exists( 'newssync_merge_new_existing' ) ) {
    /**
     * Merge novos e existentes com limite por feed.
     * Cache para evitar processamento repetido.
     */
    function newssync_merge_new_existing( $new_items, $existing, $max_per_feed = 3 ) {
        // Cache key baseado nos items
        $cache_key = 'newssync_merge_' . md5( serialize( array( $new_items, $existing, $max_per_feed ) ) );
        $cached = wp_cache_get( $cache_key, 'newssync' );

        if ( false !== $cached ) {
            return $cached;
        }

        $merged = array_merge( $new_items, $existing );

        // Remove duplicados
        if ( function_exists( 'newssync_remove_repeated_posts' ) ) {
            $merged = newssync_remove_repeated_posts( $merged );
        } else {
            $seen = array();
            $unique = array();
            foreach ( $merged as $m ) {
                $l = $m['link'] ?? '';
                if ( $l && in_array( $l, $seen, true ) ) {
                    continue;
                }
                if ( $l ) {
                    $seen[] = $l;
                }
                $unique[] = $m;
            }
            $merged = $unique;
        }

        // Agrupar por feed
        $by_feed = array();
        foreach ( $merged as $item ) {
            $key = $item['feed_id'] ?? '';
            $by_feed[ $key ][] = $item;
        }

        // Limitar por feed
        $final = array();
        foreach ( $by_feed as $feed_items ) {
            $final = array_merge( $final, array_slice( $feed_items, 0, $max_per_feed ) );
        }

        // Cache por 15 minutos
        wp_cache_set( $cache_key, $final, 'newssync', 15 * MINUTE_IN_SECONDS );

        return $final;
    }
}

if ( ! function_exists( 'newssync_create_post_from_item' ) ) {
    /**
     * Cria post a partir de item RSS.
     *
     * OTIMIZAÇÕES:
     * - Lookup via newssync_find_post_id_by_guid (indexado)
     * - Fallback para URL lookup
     * - Mapping automático GUID->post_id
     * - ZERO meta_query
     */
    function newssync_create_post_from_item( $item ) {
        $link = $item['link'] ?? '';
        if ( empty( $link ) ) {
            return false;
        }

        $post_type = 'post';

        // OTIMIZAÇÃO 1: Lookup rápido por GUID (usa cache + tabela indexada)
        $guid = sanitize_text_field( (string) ( $item['guid'] ?? $item['id'] ?? $link ) );
        $existing_id = null;

        if ( ! empty( $guid ) && function_exists( 'newssync_find_post_id_by_guid' ) ) {
            $existing_id = newssync_find_post_id_by_guid( $guid );
        }

        // OTIMIZAÇÃO 2: Se não encontrou por GUID, tentar URL
        if ( ! $existing_id && function_exists( 'newssync_find_post_id_by_original_url' ) ) {
            $existing_id = newssync_find_post_id_by_original_url( $link );
        }

        // Se existe, atualizar mapping e retornar
        if ( $existing_id ) {
            // Sincronizar GUID mapping (caso venha de fallback URL)
            if ( ! empty( $guid ) && function_exists( 'newssync_map_guid_to_post' ) ) {
                newssync_map_guid_to_post(
                    $guid,
                    intval( $existing_id ),
                    sanitize_text_field( (string) ( $item['feed_id'] ?? '' ) ) ?: null
                );
            }
            return $existing_id;
        }

        // Criar categoria (cache interno do WP já otimiza isto)
        $cat_name = sanitize_text_field( $item['category'] ?? 'rss' );
        $cat_id = get_cat_ID( $cat_name );
        if ( ! $cat_id ) {
            $cat_id = wp_create_category( $cat_name );
        }

        // Preparar conteúdo
        $source = '<hr><p><em>' . esc_html__( 'Source', 'rss-newssync' ) . ': <a href="' . esc_url( $link ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( $item['category'] ?? '' ) . '</a></em></p>';
        $content = wp_kses_post( $item['desc'] ?? '' ) . $source;

        // Timestamp robusto
        $timestamp = 0;
        if ( ! empty( $item['date_timestamp'] ) ) {
            $timestamp = (int) $item['date_timestamp'];
        } else {
            $ts = strtotime( $item['date'] ?? '' );
            if ( $ts ) {
                $timestamp = $ts;
            } else {
                $timestamp = current_time( 'timestamp' );
            }
        }

        // Prevenir timestamps futuros
        if ( $timestamp > current_time( 'timestamp' ) ) {
            $timestamp = current_time( 'timestamp' );
        }

        // Datas GMT e local
        $post_date_gmt = gmdate( 'Y-m-d H:i:s', $timestamp );
        $post_date = get_date_from_gmt( $post_date_gmt );

        // Criar post
        $post_id = wp_insert_post( array(
            'post_title'    => wp_strip_all_tags( $item['title'] ?? '' ),
            'post_content'  => $content,
            'post_status'   => get_option( 'newssync_publish_status', 'publish' ),
            'post_date'     => $post_date,
            'post_date_gmt' => $post_date_gmt,
            'post_type'     => $post_type,
            'post_author'   => get_current_user_id() ?: 1,
            'post_category' => array( $cat_id ),
        ) );

        if ( is_wp_error( $post_id ) ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled
                error_log( '[NewsSync] Erro ao criar post: ' . $post_id->get_error_message() );
            }
            return false;
        }

        // Metadados
        update_post_meta( $post_id, '_newssync_original_url', $link );
        update_post_meta( $post_id, '_newssync_source', sanitize_text_field( $item['category'] ?? '' ) );

        // ✅ NOVO: Gravar source_name (nome do site) e author (autor do artigo)
        if ( ! empty( $item['source_name'] ) ) {
            update_post_meta( $post_id, '_newssync_source_name', sanitize_text_field( $item['source_name'] ) );
        }
        if ( ! empty( $item['source_url'] ) ) {
            update_post_meta( $post_id, '_newssync_source_url', esc_url_raw( $item['source_url'] ) );
        }
        if ( ! empty( $item['author'] ) ) {
            update_post_meta( $post_id, '_newssync_author', sanitize_text_field( $item['author'] ) );
        }

        // CRÍTICO: Mapear GUID -> post_id (tabela indexada ou fallback)
        if ( ! empty( $guid ) && function_exists( 'newssync_map_guid_to_post' ) ) {
            newssync_map_guid_to_post(
                $guid,
                intval( $post_id ),
                sanitize_text_field( (string) ( $item['feed_id'] ?? '' ) ) ?: null
            );
        }

        // Thumbnail (se existir)
        if ( ! empty( $item['img_url'] ) ) {
            $local_image_url = function_exists( 'newssync_get_image_url' )
                ? newssync_get_image_url( $item['img_url'], $item['category'] ?? 'geral' )
                : '';

            if ( $local_image_url ) {
                $attach_id = attachment_url_to_postid( $local_image_url );
                if ( $attach_id ) {
                    set_post_thumbnail( $post_id, $attach_id );
                }
            }
        }

        return $post_id;
    }
}

if ( ! function_exists( 'newssync_clear_shortcode_transients' ) ) {
    /**
     * Limpar transients de shortcodes.
     * OTIMIZADO: Usa cache para lista de options, depois delete_option individual.
     */
    function newssync_clear_shortcode_transients() {
        global $wpdb;

        $patterns = array(
            '_transient_newssync_news_list_%',
            '_transient_timeout_newssync_news_list_%',
            '_transient_newssync_img_%',
            '_transient_timeout_newssync_img_%',
        );

        foreach ( $patterns as $pattern ) {
            // Cache key para esta pattern
            $cache_key = 'newssync_transient_list_' . md5( $pattern );
            $option_names = wp_cache_get( $cache_key, 'newssync' );

            if ( false === $option_names ) {
                // Query preparada para listar options
                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $sql is used below and properly prepared
                $sql = $wpdb->prepare(
                    "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
                    $pattern
                );

                // Esta query é justificada: não há API WP para listar por LIKE
                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery -- Query is prepared above, no WP API for LIKE queries on options, result is cached
                $option_names = $wpdb->get_col( $sql );

                // Cache por 5 minutos
                wp_cache_set( $cache_key, $option_names, 'newssync', 5 * MINUTE_IN_SECONDS );
            }

            // Deletar usando delete_option (respeita object-cache)
            if ( ! empty( $option_names ) && is_array( $option_names ) ) {
                foreach ( $option_names as $opt_name ) {
                    delete_option( $opt_name );
                }

                // Limpar cache desta lista
                wp_cache_delete( $cache_key, 'newssync' );
            }
        }
    }
}

if ( ! function_exists( 'newssync_import_new_posts_when_cache_expires' ) ) {
    /**
     * Import automático quando cache expira.
     *
     * OTIMIZAÇÕES:
     * - Lock com transient para evitar imports simultâneos
     * - Lookups via funções indexadas
     * - Limite de 3 posts por feed
     * - Limpa cache de shortcodes no final
     */
    function newssync_import_new_posts_when_cache_expires() {
        // Lock de import
        if ( get_transient( 'newssync_import_lock' ) ) {
            return;
        }

        $lock_duration = function_exists( 'newssync_get_cache_duration' )
            ? newssync_get_cache_duration()
            : 6 * HOUR_IN_SECONDS;

        set_transient( 'newssync_import_lock', true, $lock_duration );

        // Obter items frescos
        $fresh_items = function_exists( 'newssync_get_feed_items_raw' )
            ? newssync_get_feed_items_raw( array() )
            : array();

        if ( ! is_array( $fresh_items ) ) {
            $fresh_items = array();
        }

        $per_feed = array();

        foreach ( $fresh_items as $item ) {
            $link = $item['link'] ?? '';
            if ( empty( $link ) ) {
                continue;
            }

            // OTIMIZADO: Lookup via GUID (indexado + cache)
            $guid = sanitize_text_field( (string) ( $item['guid'] ?? $item['id'] ?? $link ) );
            $found_post_id = null;

            if ( function_exists( 'newssync_find_post_id_by_guid' ) ) {
                $found_post_id = newssync_find_post_id_by_guid( $guid );
            }

            // Fallback: URL lookup
            if ( ! $found_post_id && function_exists( 'newssync_find_post_id_by_original_url' ) ) {
                $found_post_id = newssync_find_post_id_by_original_url( $link );
            }

            // Já existe, skip
            if ( $found_post_id ) {
                continue;
            }

            // Limite por feed
            $fid = $item['feed_id'] ?? '';
            $per_feed[ $fid ] = ( $per_feed[ $fid ] ?? 0 ) + 1;
            if ( $per_feed[ $fid ] > 3 ) {
                continue;
            }

            // Criar post
            newssync_create_post_from_item( $item );
        }

        // Limpar lock e cache de shortcodes
        delete_transient( 'newssync_import_lock' );

        if ( function_exists( 'newssync_clear_shortcode_transients' ) ) {
            newssync_clear_shortcode_transients();
        }
    }
}

if ( ! function_exists( 'newssync_check_duplicate_similarity' ) ) {
    /**
     * Verifica duplicados por similaridade.
     * OTIMIZADO: Cache de posts recentes, limit 100.
     */
    function newssync_check_duplicate_similarity( $item ) {
        $threshold = (int) get_option( 'newssync_dupe_threshold', 80 );
        if ( $threshold <= 0 ) {
            $threshold = 80;
        }

        // Cache de posts recentes
        $cache_key = 'newssync_recent_posts';
        $recent_posts = wp_cache_get( $cache_key, 'newssync' );

        if ( false === $recent_posts ) {
            $recent_posts = get_posts( array(
                'post_type'      => 'post',
                'posts_per_page' => 100,
                'orderby'        => 'date',
                'order'          => 'DESC',
                'fields'         => 'ids', // Só IDs é mais rápido
            ) );

            // Cache por 15 minutos
            wp_cache_set( $cache_key, $recent_posts, 'newssync', 15 * MINUTE_IN_SECONDS );
        }

        $item_title = (string) ( $item['title'] ?? '' );

        foreach ( $recent_posts as $post_id ) {
            $post_title = get_the_title( $post_id );

            similar_text( $item_title, $post_title, $percent );

            if ( $percent >= $threshold ) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled
                    error_log( '[NewsSync] Duplicado detectado: ' . $percent . '% similaridade' );
                }
                return true;
            }
        }

        return false;
    }
}
