<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Get a server variable safely: uses filter_input() when available, falls back to $_SERVER,
 * unslashes and sanitizes the value before returning.
 *
 * @param string $key The $_SERVER key to read (e.g. 'REMOTE_ADDR', 'HTTP_USER_AGENT').
 * @return string Sanitized server value or empty string.
 */
function newssync_get_server_var( $key ) {
    if ( ! is_string( $key ) || '' === $key ) {
        return '';
    }

    $raw = null;

    if ( function_exists( 'filter_input' ) && defined( 'INPUT_SERVER' ) ) {
        if ( defined( 'FILTER_SANITIZE_FULL_SPECIAL_CHARS' ) ) {
            $raw = filter_input( INPUT_SERVER, $key, FILTER_SANITIZE_FULL_SPECIAL_CHARS );
        } elseif ( defined( 'FILTER_SANITIZE_SPECIAL_CHARS' ) ) {
            $raw = filter_input( INPUT_SERVER, $key, FILTER_SANITIZE_SPECIAL_CHARS );
        } else {
            $raw = filter_input( INPUT_SERVER, $key, FILTER_SANITIZE_STRING );
        }
    }

    if ( null === $raw ) {
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- fallback to $_SERVER
        $raw = isset( $_SERVER[ $key ] ) ? wp_unslash( $_SERVER[ $key ] ) : '';
    } else {
        $raw = (string) $raw;
        $raw = wp_unslash( $raw );
    }

    if ( 'REMOTE_ADDR' === $key ) {
        $ip = filter_var( $raw, FILTER_VALIDATE_IP );
        return $ip ? (string) $ip : '';
    }

    $sanitized = sanitize_text_field( (string) $raw );

    return $sanitized;
}

/**
 * Get cache duration in seconds.
 *
 * Priority:
 * 1. Use $custom_ttl if provided (from shortcode cache_ttl parameter)
 *    - "0" or 0 = disable cache (1 second only, forces refresh)
 * 2. Read `newssync_cache_duration` option (string like '6h' or '24h') and map to seconds.
 * 3. Fallback to 6 hours.
 *
 * @param string|int $custom_ttl Optional custom TTL from shortcode (e.g. '30m', '2h', 3600, "0")
 * @return int Seconds to use for transients/cache.
 */
if ( ! function_exists( 'newssync_get_cache_duration' ) ) {
    function newssync_get_cache_duration( $custom_ttl = '' ) {
        // Priority 1: custom TTL from shortcode (check if explicitly provided, even if "0")
        if ( $custom_ttl !== '' && $custom_ttl !== null ) {
            // If numeric, treat as seconds directly
            if ( is_numeric( $custom_ttl ) ) {
                $seconds = intval( $custom_ttl );
                // Special case: 0 = disable cache (1 second to force refresh)
                if ( $seconds === 0 ) {
                    return 1;
                }
                return max( 60, $seconds ); // minimum 1 minute for non-zero values
            }

            // Parse string format (e.g. '30m', '2h', '1d')
            $val = trim( (string) $custom_ttl );
            if ( preg_match( '/^(\d+)([mhd])$/i', $val, $m ) ) {
                $num = intval( $m[1] );
                $unit = strtolower( $m[2] );

                if ( 'm' === $unit ) {
                    return max( 60, $num * 60 );
                } elseif ( 'h' === $unit ) {
                    return max( 60, $num * 3600 );
                } elseif ( 'd' === $unit ) {
                    return max( 60, $num * 86400 );
                }
            }
        }

        // Priority 2: option setting
        $opt = get_option( 'newssync_cache_duration', '6h' );

        $map = [
            '6h'  => 6 * HOUR_IN_SECONDS,
            '24h' => 24 * HOUR_IN_SECONDS,
        ];

        if ( isset( $map[ $opt ] ) ) {
            return (int) $map[ $opt ];
        }

        return 6 * HOUR_IN_SECONDS;
    }
}

/**
 * Wrapper for get_option with a default.
 *
 * @param string $key Option name
 * @param mixed $default Default value if option not set
 * @return mixed
 */
function newssync_get_option( $key, $default = null ) {
    if ( ! is_string( $key ) || $key === '' ) {
        return $default;
    }
    $val = get_option( $key, $default );
    return $val;
}
