<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Activation/deactivation hooks para NewsSync.
 *
 * Incluir no ficheiro principal do plugin:
 * require_once plugin_dir_path( __FILE__ ) . 'includes/core/newssync-activation.php';
 * register_activation_hook( __FILE__, 'newssync_plugin_activate' );
 * register_deactivation_hook( __FILE__, 'newssync_plugin_deactivate' );
 */

if ( ! function_exists( 'newssync_plugin_activate' ) ) {
    /**
     * Hook de ativação: tenta criar tabela otimizada.
     * Se falhar, plugin continua funcional com fallback.
     */
    function newssync_plugin_activate() {
        // ✅ MELHOR:  caminho relativo ao próprio arquivo
        $db_file = __DIR__ . '/newssync-db.php';  // Mesmo diretório que activation

        if ( file_exists( $db_file ) ) {
            require_once $db_file;
        } else {
            // Debug helper
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled
                error_log( '[NewsSync] ERRO: newssync-db.php não encontrado em:  ' . $db_file );
            }
        }

        // Tentar criar tabela
        if ( function_exists( 'newssync_create_item_map_table' ) ) {
            $result = newssync_create_item_map_table();

            if ( $result['success'] ) {
                // Sucesso: marcar flag
                update_option( 'newssync_optimized_table_available', 1 );
                delete_option( 'newssync_last_table_error' );

                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled
                    error_log( '[NewsSync] Plugin ativado com tabela otimizada' );
                }
            } else {
                // Falhou:  guardar erro mas não bloquear ativação
                update_option( 'newssync_optimized_table_available', 0 );

                // CORREÇÃO: só guardar se realmente houver erro
                if ( !  empty( $result['error'] ) ) {
                    update_option( 'newssync_last_table_error', substr( $result['error'], 0, 2000 ) );
                } else {
                    update_option( 'newssync_last_table_error', 'Unknown error during table creation' );
                }

                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    $error_msg = !  empty( $result['error'] ) ? $result['error'] : 'unknown reason';
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled
                    error_log( '[NewsSync] Plugin ativado SEM tabela otimizada (fallback ativo): ' . $error_msg );
                }
            }
        } else {
            // Função não disponível
            update_option( 'newssync_optimized_table_available', 0 );
            update_option( 'newssync_last_table_error', 'DB helper function not loaded during activation' );
        }

        // Criar opções default se não existirem
        add_option( 'newssync_default_layout', 'grid' );
        add_option( 'newssync_cache_duration', '6h' );
        add_option( 'newssync_show_image', 1 );
        add_option( 'newssync_save_as_posts', 0 );
        add_option( 'newssync_publish_status', 'publish' );
        add_option( 'newssync_max_import_items', 15 );
        add_option( 'newssync_dupe_threshold', 80 );
        add_option( 'newssync_feeds', array() );

        // Flush rewrite rules
        flush_rewrite_rules();
    }
}

if ( ! function_exists( 'newssync_plugin_deactivate' ) ) {
    /**
     * Hook de desativação: limpar transients e flush rules.
     * NÃO apaga a tabela nem as opções (preservar dados).
     */
    function newssync_plugin_deactivate() {
        // Limpar todos os transients do plugin
        global $wpdb;

        $patterns = array(
            '_transient_newssync_%',
            '_transient_timeout_newssync_%',
        );

        foreach ( $patterns as $pattern ) {
            $sql = $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
                $pattern
            );

            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Query is prepared above, cleanup operation on deactivation, no caching needed
            $options = $wpdb->get_col( $sql );

            if ( ! empty( $options ) ) {
                foreach ( $options as $option ) {
                    delete_option( $option );
                }
            }
        }

        // Flush rewrite rules
        flush_rewrite_rules();

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled
            error_log( '[NewsSync] Plugin desativado, cache limpo' );
        }
    }
}

if ( ! function_exists( 'newssync_check_table_on_admin_init' ) ) {
    /**
     * Verificar tabela em admin_init (só se ainda não existir).
     * Tenta criar automaticamente se possível.
     */
    add_action( 'admin_init', 'newssync_check_table_on_admin_init' );

    function newssync_check_table_on_admin_init() {
        // Só executar para admins
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Verificar se já checámos recentemente
        if ( get_transient( 'newssync_table_check_done' ) ) {
            return;
        }

        // Marcar como checked (1 hora)
        set_transient( 'newssync_table_check_done', true, HOUR_IN_SECONDS );

        // Se tabela existe, tudo OK
        if ( function_exists( 'newssync_table_exists' ) && newssync_table_exists() ) {
            update_option( 'newssync_optimized_table_available', 1 );
            delete_option( 'newssync_last_table_error' ); // ← ADICIONAR ESTA LINHA
            return;
        }

        // Tabela não existe: tentar criar silenciosamente
        if ( function_exists( 'newssync_create_item_map_table' ) ) {
            $result = newssync_create_item_map_table();

            if ( $result['success'] ) {
                update_option( 'newssync_optimized_table_available', 1 );
                delete_option( 'newssync_last_table_error' );
            } else {
                update_option( 'newssync_optimized_table_available', 0 );

                if ( ! empty( $result['error'] ) ) {
                    update_option( 'newssync_last_table_error', substr( $result['error'], 0, 2000 ) );
                }
            }
        }
    }
}
