<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * WP-CLI command to migrate historical _newssync_guid postmeta into the indexed table.
 * Usage: wp newssync migrate-item-map --batch=200
 */
if ( defined( 'WP_CLI' ) && WP_CLI ) {
    class Newssync_Migrate_Item_Map_Command {
        public function migrate( $args, $assoc_args ) {
            $batch = isset( $assoc_args['batch'] ) ? absint( $assoc_args['batch'] ) : 200;
            if ( $batch <= 0 ) $batch = 200;

            if ( ! function_exists( 'newssync_table_exists' ) ) {
                WP_CLI::error( 'newssync_table_exists() not available. Ensure includes/core/newssync-db.php is loaded.' );
                return;
            }
            if ( ! newssync_table_exists() ) {
                if ( ! function_exists( 'newssync_create_item_map_table' ) ) {
                    WP_CLI::error( 'newssync_create_item_map_table() not available.' );
                    return;
                }
                WP_CLI::log( 'Creating newssync_item_map table...' );
                newssync_create_item_map_table();
                if ( ! newssync_table_exists() ) {
                    WP_CLI::error( 'Failed to create newssync_item_map table.' );
                    return;
                }
            }

            $paged = 1;
            $total_migrated = 0;
            do {
                global $wpdb;
                $offset = ( $paged - 1 ) * $batch;

                // Direct query to fetch post IDs with _newssync_guid meta (faster than WP_Query with meta_query)
                $sql = $wpdb->prepare(
                    "SELECT DISTINCT post_id
                    FROM {$wpdb->postmeta}
                    WHERE meta_key = %s
                    ORDER BY post_id ASC
                    LIMIT %d OFFSET %d",
                    '_newssync_guid',
                    $batch,
                    $offset
                );

                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- WP-CLI migration command, query is prepared above, no caching needed for one-time migration
                $post_ids = $wpdb->get_col( $sql );

                if ( empty( $post_ids ) ) {
                    break;
                }

                foreach ( $post_ids as $post_id ) {
                    $guid = get_post_meta( $post_id, '_newssync_guid', true );
                    $guid = sanitize_text_field( (string) $guid );
                    if ( '' === $guid ) continue;
                    if ( function_exists( 'newssync_map_guid_to_post' ) ) {
                        newssync_map_guid_to_post( $guid, intval( $post_id ) );
                        $total_migrated++;
                    }
                }

                WP_CLI::log( "Migrated batch {$paged} (total migrated: {$total_migrated})" );
                $paged++;
                wp_reset_postdata();
            } while ( true );

            WP_CLI::success( "Migration finished. Total migrated: {$total_migrated}" );
        }
    }

    WP_CLI::add_command( 'newssync migrate-item-map', array( 'Newssync_Migrate_Item_Map_Command', 'migrate' ) );
}
